<?php
/**
 * Trading Bot Cron Job
 * This script should be run every minute via cron
 * 
 * Cron setup:
 * * * * * * /usr/bin/php /home/username/public_html/trading-bot/cron/trading_bot.php >> /home/username/public_html/trading-bot/logs/cron.log 2>&1
 */

require_once __DIR__ . '/../includes/TradingEngine.php';
require_once __DIR__ . '/../includes/Database.php';

// Set unlimited execution time
set_time_limit(0);

// Log start
$logFile = __DIR__ . '/../logs/cron.log';
$startTime = date('Y-m-d H:i:s');
file_put_contents($logFile, "[$startTime] Cron job started\n", FILE_APPEND);

try {
    $db = Database::getInstance();
    
    // Get all active users with trading enabled
    $sql = "SELECT ts.user_id 
            FROM trading_settings ts 
            WHERE ts.is_active = 1";
    
    $activeUsers = $db->fetchAll($sql);
    
    if (empty($activeUsers)) {
        file_put_contents($logFile, "[$startTime] No active users found\n", FILE_APPEND);
        exit(0);
    }
    
    foreach ($activeUsers as $user) {
        $userId = $user['user_id'];
        $userStartTime = microtime(true);
        
        try {
            // Initialize trading engine for this user
            $tradingEngine = new TradingEngine($userId);
            
            // Run trading logic
            $result = $tradingEngine->run();
            
            $executionTime = round(microtime(true) - $userStartTime, 3);
            
            // Log result
            $logMessage = sprintf(
                "[%s] User %d: %s (%.3fs)\n",
                date('Y-m-d H:i:s'),
                $userId,
                json_encode($result),
                $executionTime
            );
            
            file_put_contents($logFile, $logMessage, FILE_APPEND);
            
        } catch (Exception $e) {
            $errorMessage = sprintf(
                "[%s] User %d ERROR: %s\n",
                date('Y-m-d H:i:s'),
                $userId,
                $e->getMessage()
            );
            
            file_put_contents($logFile, $errorMessage, FILE_APPEND);
            
            // Log to database
            $db->insert('bot_logs', [
                'user_id' => $userId,
                'level' => 'error',
                'message' => $e->getMessage(),
                'context' => 'cron_job'
            ]);
        }
    }
    
    // Update last cron run
    $db->update('system_settings',
        ['setting_value' => date('Y-m-d H:i:s')],
        'setting_key = :key',
        ['key' => 'last_cron_run']
    );
    
    $endTime = date('Y-m-d H:i:s');
    file_put_contents($logFile, "[$endTime] Cron job completed\n", FILE_APPEND);
    
} catch (Exception $e) {
    $errorTime = date('Y-m-d H:i:s');
    file_put_contents($logFile, "[$errorTime] CRITICAL ERROR: " . $e->getMessage() . "\n", FILE_APPEND);
    exit(1);
}

exit(0);
