<?php
/**
 * Trading Engine Class
 * Handles Grid Trading and DCA (Averaging) logic
 */

require_once __DIR__ . '/BitgetAPI.php';
require_once __DIR__ . '/Telegram.php';
require_once __DIR__ . '/Database.php';

class TradingEngine {
    private $db;
    private $api;
    private $telegram;
    private $userId;
    private $settings;
    private $lastError;
    private $currentBalance;

    public function __construct($userId) {
        $this->db = Database::getInstance();
        $this->userId = $userId;
        $this->loadSettings();
        $this->initializeAPI();
        $this->initializeTelegram();
    }

    /**
     * Load user settings
     */
    private function loadSettings() {
        $sql = "SELECT ts.*, ap.api_key, ap.api_secret, ap.passphrase, ap.is_testnet 
                FROM trading_settings ts 
                LEFT JOIN api_settings ap ON ts.user_id = ap.user_id 
                WHERE ts.user_id = ? AND ap.is_active = 1 
                LIMIT 1";
        
        $this->settings = $this->db->fetch($sql, [$this->userId]);
        
        if (!$this->settings) {
            throw new Exception("Trading settings not found for user");
        }
    }

    /**
     * Initialize Bitget API
     */
    private function initializeAPI() {
        if (empty($this->settings['api_key'])) {
            throw new Exception("API credentials not configured");
        }

        $this->api = new BitgetAPI(
            $this->settings['api_key'],
            $this->settings['api_secret'],
            $this->settings['passphrase'],
            $this->settings['is_testnet']
        );
    }

    /**
     * Initialize Telegram
     */
    private function initializeTelegram() {
        $sql = "SELECT * FROM telegram_settings WHERE user_id = ? LIMIT 1";
        $telegramSettings = $this->db->fetch($sql, [$this->userId]);

        if ($telegramSettings) {
            $this->telegram = new Telegram(
                $telegramSettings['bot_token'],
                $telegramSettings['chat_id'],
                $telegramSettings['enabled']
            );
        } else {
            $this->telegram = new Telegram();
        }
    }

    /**
     * Get current balance
     */
    public function getBalance() {
        if ($this->settings['is_test_mode']) {
            return $this->settings['test_balance'];
        }

        $balance = $this->api->getBalance('USDT');
        if ($balance && isset($balance[0]['available'])) {
            $this->currentBalance = (float) $balance[0]['available'];
            return $this->currentBalance;
        }

        return 0;
    }

    /**
     * Main trading loop
     */
    public function run() {
        try {
            // Check if bot is active
            if (!$this->settings['is_active']) {
                return ['status' => 'inactive', 'message' => 'Bot is not active'];
            }

            // Get current balance
            $balance = $this->getBalance();
            if ($balance <= 0) {
                return ['status' => 'error', 'message' => 'Insufficient balance'];
            }

            // Check risk management
            if (!$this->checkRiskManagement()) {
                return ['status' => 'stopped', 'message' => 'Risk management triggered'];
            }

            $results = [];

            // Run Grid Trading
            if ($this->settings['mode_grid']) {
                $gridResult = $this->runGridTrading();
                $results['grid'] = $gridResult;
            }

            // Run DCA Trading
            if ($this->settings['mode_dca']) {
                $dcaResult = $this->runDCATrading();
                $results['dca'] = $dcaResult;
            }

            // Check take profit
            $this->checkTakeProfit();

            // Update daily stats
            $this->updateDailyStats();

            return ['status' => 'success', 'results' => $results];

        } catch (Exception $e) {
            $this->lastError = $e->getMessage();
            $this->log('error', $e->getMessage());
            
            if ($this->telegram) {
                $this->telegram->notifyError($e->getMessage(), 'Trading Engine');
            }

            return ['status' => 'error', 'message' => $e->getMessage()];
        }
    }

    /**
     * Grid Trading Logic
     */
    private function runGridTrading() {
        $symbol = $this->settings['symbol'];
        $currentPrice = $this->api->getCurrentPrice($symbol);
        
        if (!$currentPrice) {
            return ['error' => 'Could not get current price'];
        }

        // Dynamic grid adjustment
        $upperPrice = $this->settings['grid_upper_price'];
        $lowerPrice = $this->settings['grid_lower_price'];
        
        if ($this->settings['grid_dynamic']) {
            // Adjust grid based on volatility
            $klines = $this->api->getKlines($symbol, $this->settings['timeframe'], 20);
            if ($klines) {
                $prices = array_column($klines, 'close');
                $volatility = $this->calculateVolatility($prices);
                
                // Adjust grid range based on volatility
                $range = $currentPrice * ($volatility * 2);
                $upperPrice = $currentPrice + $range;
                $lowerPrice = $currentPrice - $range;
            }
        }

        // If grid prices not set, use default range
        if ($upperPrice == 0 || $lowerPrice == 0) {
            $upperPrice = $currentPrice * 1.05; // 5% above
            $lowerPrice = $currentPrice * 0.95; // 5% below
        }

        // Calculate grid levels
        $gridLevels = $this->api->calculateGridLevels(
            $lowerPrice, 
            $upperPrice, 
            $this->settings['grid_levels']
        );

        // Get existing grid orders
        $existingOrders = $this->getExistingGridOrders();
        
        $results = [
            'price' => $currentPrice,
            'levels_created' => 0,
            'levels_filled' => 0,
            'orders' => []
        ];

        // Create or update grid orders
        foreach ($gridLevels as $index => $price) {
            $orderExists = false;
            
            foreach ($existingOrders as $order) {
                if (abs($order['price'] - $price) < 0.0001) {
                    $orderExists = true;
                    break;
                }
            }

            if (!$orderExists) {
                // Determine side based on price relative to current price
                $side = $price < $currentPrice ? 'buy' : 'sell';
                
                // Calculate quantity for this grid level
                $investmentPerGrid = $this->settings['grid_investment'] / $this->settings['grid_levels'];
                $quantity = $investmentPerGrid / $price;
                
                // Place grid order
                $orderResult = $this->placeGridOrder($symbol, $side, $price, $quantity, $index);
                
                if ($orderResult) {
                    $results['levels_created']++;
                    $results['orders'][] = $orderResult;
                }
            }
        }

        // Check filled orders and create opposite orders
        $this->processFilledGridOrders();

        return $results;
    }

    /**
     * DCA (Averaging) Trading Logic
     */
    private function runDCATrading() {
        $symbol = $this->settings['symbol'];
        $currentPrice = $this->api->getCurrentPrice($symbol);
        
        if (!$currentPrice) {
            return ['error' => 'Could not get current price'];
        }

        // Get existing DCA orders
        $existingOrders = $this->getExistingDCAOrders();
        
        // Get active DCA position
        $activePosition = $this->getActiveDCAPosition();
        
        $results = [
            'price' => $currentPrice,
            'levels_created' => 0,
            'levels_triggered' => 0,
            'orders' => []
        ];

        if (!$activePosition) {
            // No active position, create initial DCA setup
            $initialQuantity = $this->calculateDCAQuantity(1);
            
            // Place initial buy order
            $orderResult = $this->api->placeOrder(
                $symbol,
                'buy',
                'market',
                $initialQuantity
            );

            if ($orderResult) {
                $this->recordTrade('buy', $symbol, $currentPrice, $initialQuantity, 'dca', 1);
                $results['levels_created']++;
                
                // Create DCA levels
                $this->createDCALevels($symbol, $currentPrice);
            }
        } else {
            // Check if any DCA level should be triggered
            foreach ($existingOrders as $order) {
                if ($order['status'] === 'pending' && $currentPrice <= $order['trigger_price']) {
                    // Trigger DCA buy
                    $orderResult = $this->api->placeOrder(
                        $symbol,
                        'buy',
                        'market',
                        $order['quantity']
                    );

                    if ($orderResult) {
                        $this->updateDCAOrder($order['id'], 'filled');
                        $this->recordTrade('buy', $symbol, $currentPrice, $order['quantity'], 'dca', $order['dca_level']);
                        $results['levels_triggered']++;
                        
                        // Notify
                        if ($this->telegram) {
                            $total = $currentPrice * $order['quantity'];
                            $this->telegram->notifyTrade('DCA', $symbol, 'buy', $currentPrice, $order['quantity'], $total);
                        }
                    }
                }
            }
        }

        return $results;
    }

    /**
     * Create DCA levels
     */
    private function createDCALevels($symbol, $currentPrice) {
        $dropPercentage = $this->settings['dca_drop_percentage'] / 100;
        $multiplier = $this->settings['dca_multiplier'];
        
        for ($i = 2; $i <= $this->settings['dca_levels']; $i++) {
            $triggerPrice = $currentPrice * pow(1 - $dropPercentage, $i - 1);
            $quantity = $this->calculateDCAQuantity($i);
            
            $this->db->insert('dca_orders', [
                'user_id' => $this->userId,
                'symbol' => $symbol,
                'dca_level' => $i,
                'trigger_price' => $triggerPrice,
                'quantity' => $quantity,
                'multiplier' => pow($multiplier, $i - 1),
                'status' => 'pending'
            ]);
        }
    }

    /**
     * Calculate DCA quantity based on level
     */
    private function calculateDCAQuantity($level) {
        $baseInvestment = $this->settings['dca_max_investment'] / $this->settings['dca_levels'];
        $multiplier = $this->settings['dca_multiplier'];
        
        return $baseInvestment * pow($multiplier, $level - 1);
    }

    /**
     * Place grid order
     */
    private function placeGridOrder($symbol, $side, $price, $quantity, $level) {
        // Check max positions
        if ($this->settings['risk_max_positions_enabled']) {
            $openPositions = $this->getOpenPositionsCount();
            if ($openPositions >= $this->settings['risk_max_positions']) {
                return false;
            }
        }

        $orderResult = $this->api->placeOrder(
            $symbol,
            $side,
            'limit',
            $quantity,
            $price
        );

        if ($orderResult && isset($orderResult['orderId'])) {
            // Save to database
            $this->db->insert('grid_orders', [
                'user_id' => $this->userId,
                'symbol' => $symbol,
                'grid_level' => $level,
                'price' => $price,
                'quantity' => $quantity,
                'side' => $side,
                'status' => 'pending',
                'order_id' => $orderResult['orderId']
            ]);

            return $orderResult;
        }

        return false;
    }

    /**
     * Process filled grid orders
     */
    private function processFilledGridOrders() {
        $openOrders = $this->getExistingGridOrders();
        
        foreach ($openOrders as $order) {
            if ($order['status'] !== 'filled') {
                // Check order status from API
                $orderInfo = $this->api->getOrder($order['symbol'], $order['order_id']);
                
                if ($orderInfo && isset($orderInfo['status']) && $orderInfo['status'] === 'filled') {
                    // Update order status
                    $this->db->update('grid_orders', 
                        ['status' => 'filled', 'filled_at' => date('Y-m-d H:i:s')],
                        'id = :id',
                        ['id' => $order['id']]
                    );

                    // Record trade
                    $this->recordTrade(
                        $order['side'],
                        $order['symbol'],
                        $order['price'],
                        $order['quantity'],
                        'grid',
                        $order['grid_level']
                    );

                    // Create opposite order
                    $oppositeSide = $order['side'] === 'buy' ? 'sell' : 'buy';
                    $oppositePrice = $order['side'] === 'buy' 
                        ? $order['price'] * (1 + ($this->settings['take_profit_percentage'] / 100))
                        : $order['price'] * (1 - ($this->settings['take_profit_percentage'] / 100));

                    $this->placeGridOrder(
                        $order['symbol'],
                        $oppositeSide,
                        $oppositePrice,
                        $order['quantity'],
                        $order['grid_level']
                    );

                    // Notify
                    if ($this->telegram) {
                        $total = $order['price'] * $order['quantity'];
                        $this->telegram->notifyTrade('Grid', $order['symbol'], $order['side'], $order['price'], $order['quantity'], $total);
                    }
                }
            }
        }
    }

    /**
     * Check take profit conditions
     */
    private function checkTakeProfit() {
        if (!$this->settings['take_profit_enabled']) {
            return;
        }

        $activeTrades = $this->getActiveTrades();
        $currentPrice = $this->api->getCurrentPrice($this->settings['symbol']);
        
        foreach ($activeTrades as $trade) {
            if ($trade['side'] === 'buy') {
                $profitPercentage = (($currentPrice - $trade['entry_price']) / $trade['entry_price']) * 100;
                
                if ($profitPercentage >= $this->settings['take_profit_percentage']) {
                    // Close position
                    $this->closePosition($trade, $currentPrice);
                }
            }
        }
    }

    /**
     * Close position
     */
    private function closePosition($trade, $exitPrice) {
        $symbol = $trade['symbol'];
        $quantity = $trade['quantity'];
        
        // Place sell order
        $orderResult = $this->api->placeOrder($symbol, 'sell', 'market', $quantity);
        
        if ($orderResult) {
            $profit = ($exitPrice - $trade['entry_price']) * $quantity;
            $profitPercentage = (($exitPrice - $trade['entry_price']) / $trade['entry_price']) * 100;
            
            // Update trade record
            $this->db->update('active_trades',
                ['status' => 'closed', 'closed_at' => date('Y-m-d H:i:s')],
                'id = :id',
                ['id' => $trade['id']]
            );

            // Add to history
            $this->db->insert('trade_history', [
                'user_id' => $this->userId,
                'symbol' => $symbol,
                'side' => 'sell',
                'entry_price' => $trade['entry_price'],
                'exit_price' => $exitPrice,
                'quantity' => $quantity,
                'total_amount' => $exitPrice * $quantity,
                'profit_loss' => $profit,
                'profit_loss_percentage' => $profitPercentage,
                'trade_type' => $trade['trade_type'],
                'status' => $profit >= 0 ? 'win' : 'loss',
                'closed_at' => date('Y-m-d H:i:s')
            ]);

            // Notify
            if ($this->telegram) {
                $balance = $this->getBalance();
                $this->telegram->notifyProfit($symbol, $profit, $profitPercentage, $balance);
            }
        }
    }

    /**
     * Risk Management checks
     */
    private function checkRiskManagement() {
        // Check stop loss
        if ($this->settings['risk_stop_loss_enabled']) {
            $dailyStats = $this->getTodayStats();
            if ($dailyStats && $dailyStats['total_loss'] >= ($this->settings['live_balance'] * ($this->settings['risk_stop_loss_percentage'] / 100))) {
                $this->log('warning', 'Stop loss triggered');
                $this->stopBot('Stop loss triggered');
                return false;
            }
        }

        // Check max drawdown
        if ($this->settings['risk_max_drawdown_enabled']) {
            $drawdown = $this->calculateDrawdown();
            if ($drawdown >= $this->settings['risk_max_drawdown_percentage']) {
                $this->log('warning', 'Max drawdown reached: ' . $drawdown . '%');
                $this->stopBot('Max drawdown reached');
                return false;
            }
        }

        // Check daily loss limit
        if ($this->settings['risk_daily_loss_enabled']) {
            $dailyStats = $this->getTodayStats();
            if ($dailyStats && abs($dailyStats['net_profit']) >= $this->settings['risk_daily_loss_amount']) {
                $this->log('warning', 'Daily loss limit reached');
                $this->stopBot('Daily loss limit reached');
                return false;
            }
        }

        return true;
    }

    /**
     * Calculate drawdown
     */
    private function calculateDrawdown() {
        $sql = "SELECT MAX(balance_end) as peak FROM daily_stats WHERE user_id = ?";
        $result = $this->db->fetch($sql, [$this->userId]);
        
        if (!$result || !$result['peak']) {
            return 0;
        }

        $peak = $result['peak'];
        $current = $this->getBalance();
        
        return (($peak - $current) / $peak) * 100;
    }

    /**
     * Calculate volatility
     */
    private function calculateVolatility($prices) {
        $returns = [];
        for ($i = 1; $i < count($prices); $i++) {
            $returns[] = ($prices[$i] - $prices[$i-1]) / $prices[$i-1];
        }
        
        $mean = array_sum($returns) / count($returns);
        $variance = 0;
        
        foreach ($returns as $return) {
            $variance += pow($return - $mean, 2);
        }
        
        $variance /= count($returns);
        return sqrt($variance);
    }

    /**
     * Record trade
     */
    private function recordTrade($side, $symbol, $price, $quantity, $type, $level = null) {
        return $this->db->insert('active_trades', [
            'user_id' => $this->userId,
            'symbol' => $symbol,
            'side' => $side,
            'entry_price' => $price,
            'quantity' => $quantity,
            'total_amount' => $price * $quantity,
            'trade_type' => $type,
            'grid_level' => $type === 'grid' ? $level : null,
            'dca_level' => $type === 'dca' ? $level : null,
            'status' => 'open'
        ]);
    }

    /**
     * Update daily statistics
     */
    private function updateDailyStats() {
        $today = date('Y-m-d');
        $balance = $this->getBalance();
        
        // Check if record exists
        $sql = "SELECT * FROM daily_stats WHERE user_id = ? AND date = ?";
        $existing = $this->db->fetch($sql, [$this->userId, $today]);
        
        if ($existing) {
            $this->db->update('daily_stats',
                ['balance_end' => $balance, 'updated_at' => date('Y-m-d H:i:s')],
                'id = :id',
                ['id' => $existing['id']]
            );
        } else {
            $this->db->insert('daily_stats', [
                'user_id' => $this->userId,
                'date' => $today,
                'balance_start' => $balance,
                'balance_end' => $balance
            ]);
        }
    }

    /**
     * Get today's stats
     */
    private function getTodayStats() {
        $sql = "SELECT * FROM daily_stats WHERE user_id = ? AND date = ?";
        return $this->db->fetch($sql, [$this->userId, date('Y-m-d')]);
    }

    /**
     * Get active trades
     */
    private function getActiveTrades() {
        $sql = "SELECT * FROM active_trades WHERE user_id = ? AND status = 'open'";
        return $this->db->fetchAll($sql, [$this->userId]);
    }

    /**
     * Get open positions count
     */
    private function getOpenPositionsCount() {
        $sql = "SELECT COUNT(*) as count FROM active_trades WHERE user_id = ? AND status = 'open'";
        $result = $this->db->fetch($sql, [$this->userId]);
        return $result ? $result['count'] : 0;
    }

    /**
     * Get existing grid orders
     */
    private function getExistingGridOrders() {
        $sql = "SELECT * FROM grid_orders WHERE user_id = ? AND status = 'pending'";
        return $this->db->fetchAll($sql, [$this->userId]);
    }

    /**
     * Get existing DCA orders
     */
    private function getExistingDCAOrders() {
        $sql = "SELECT * FROM dca_orders WHERE user_id = ? AND status = 'pending'";
        return $this->db->fetchAll($sql, [$this->userId]);
    }

    /**
     * Get active DCA position
     */
    private function getActiveDCAPosition() {
        $sql = "SELECT * FROM active_trades WHERE user_id = ? AND trade_type = 'dca' AND status = 'open' ORDER BY created_at DESC LIMIT 1";
        return $this->db->fetch($sql, [$this->userId]);
    }

    /**
     * Update DCA order
     */
    private function updateDCAOrder($orderId, $status) {
        $this->db->update('dca_orders',
            ['status' => $status, 'filled_at' => date('Y-m-d H:i:s')],
            'id = :id',
            ['id' => $orderId]
        );
    }

    /**
     * Stop bot
     */
    private function stopBot($reason) {
        $this->db->update('trading_settings',
            ['is_active' => 0],
            'user_id = :user_id',
            ['user_id' => $this->userId]
        );

        if ($this->telegram) {
            $this->telegram->notifyStatus('stopped', ['reason' => $reason]);
        }
    }

    /**
     * Log message
     */
    private function log($level, $message, $context = '') {
        $this->db->insert('bot_logs', [
            'user_id' => $this->userId,
            'level' => $level,
            'message' => $message,
            'context' => $context
        ]);
    }

    /**
     * Get last error
     */
    public function getLastError() {
        return $this->lastError;
    }

    /**
     * Get settings
     */
    public function getSettings() {
        return $this->settings;
    }
}
