<?php
/**
 * Main Dashboard
 * Bitget Trading Bot
 */

require_once __DIR__ . '/../includes/Auth.php';
require_once __DIR__ . '/../includes/Database.php';

$auth = Auth::getInstance();
$auth->requireAuth();

$db = Database::getInstance();
$userId = $auth->getUserId();

// Get user settings
$settings = $db->fetch("SELECT * FROM trading_settings WHERE user_id = ?", [$userId]);

// Get today's stats
$todayStats = $db->fetch("SELECT * FROM daily_stats WHERE user_id = ? AND date = ?", [$userId, date('Y-m-d')]);

// Get active trades
$activeTrades = $db->fetchAll("SELECT * FROM active_trades WHERE user_id = ? AND status = 'open' ORDER BY created_at DESC LIMIT 10", [$userId]);

// Get recent trades
$recentTrades = $db->fetchAll("SELECT * FROM trade_history WHERE user_id = ? ORDER BY created_at DESC LIMIT 10", [$userId]);

// Get recent logs
$recentLogs = $db->fetchAll("SELECT * FROM bot_logs WHERE user_id = ? ORDER BY created_at DESC LIMIT 5", [$userId]);

// Calculate stats
$totalTrades = $db->fetch("SELECT COUNT(*) as count FROM trade_history WHERE user_id = ?", [$userId]);
$winningTrades = $db->fetch("SELECT COUNT(*) as count FROM trade_history WHERE user_id = ? AND status = 'win'", [$userId]);
$totalProfit = $db->fetch("SELECT SUM(profit_loss) as total FROM trade_history WHERE user_id = ?", [$userId]);

$winRate = $totalTrades['count'] > 0 ? round(($winningTrades['count'] / $totalTrades['count']) * 100, 2) : 0;
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard - Bitget Trading Bot</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <style>
        @import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');
        body { font-family: 'Inter', sans-serif; }
        .gradient-bg { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); }
        .glass-effect { background: rgba(255, 255, 255, 0.05); backdrop-filter: blur(10px); }
        .card-hover { transition: all 0.3s ease; }
        .card-hover:hover { transform: translateY(-5px); box-shadow: 0 20px 40px rgba(0,0,0,0.1); }
        .status-dot { animation: pulse 2s infinite; }
        @keyframes pulse {
            0%, 100% { opacity: 1; }
            50% { opacity: 0.5; }
        }
        .sidebar-link { transition: all 0.3s ease; }
        .sidebar-link:hover { background: rgba(99, 102, 241, 0.1); color: #6366f1; }
        .sidebar-link.active { background: rgba(99, 102, 241, 0.1); color: #6366f1; border-right: 3px solid #6366f1; }
    </style>
</head>
<body class="bg-gray-50">
    <div class="flex h-screen">
        <!-- Sidebar -->
        <aside class="w-64 bg-white shadow-xl">
            <div class="p-6">
                <div class="flex items-center space-x-3">
                    <div class="w-10 h-10 gradient-bg rounded-lg flex items-center justify-center">
                        <i class="fas fa-robot text-white text-xl"></i>
                    </div>
                    <div>
                        <h1 class="text-xl font-bold text-gray-800">Lompodoe</h1>
                        <p class="text-xs text-gray-500">Trading Bot</p>
                    </div>
                </div>
            </div>
            
            <nav class="mt-6">
                <a href="index.php" class="sidebar-link active flex items-center px-6 py-3 text-gray-700">
                    <i class="fas fa-home w-6"></i>
                    <span>Dashboard</span>
                </a>
                <a href="trading.php" class="sidebar-link flex items-center px-6 py-3 text-gray-700">
                    <i class="fas fa-chart-line w-6"></i>
                    <span>Trading</span>
                </a>
                <a href="settings.php" class="sidebar-link flex items-center px-6 py-3 text-gray-700">
                    <i class="fas fa-cog w-6"></i>
                    <span>Settings</span>
                </a>
                <a href="history.php" class="sidebar-link flex items-center px-6 py-3 text-gray-700">
                    <i class="fas fa-history w-6"></i>
                    <span>History</span>
                </a>
                <?php if ($auth->isAdmin()): ?>
                <a href="users.php" class="sidebar-link flex items-center px-6 py-3 text-gray-700">
                    <i class="fas fa-users w-6"></i>
                    <span>Users</span>
                </a>
                <?php endif; ?>
                <a href="logout.php" class="sidebar-link flex items-center px-6 py-3 text-gray-700">
                    <i class="fas fa-sign-out-alt w-6"></i>
                    <span>Logout</span>
                </a>
            </nav>
        </aside>

        <!-- Main Content -->
        <main class="flex-1 overflow-y-auto">
            <!-- Header -->
            <header class="bg-white shadow-sm">
                <div class="flex items-center justify-between px-8 py-4">
                    <div>
                        <h2 class="text-2xl font-bold text-gray-800">Dashboard</h2>
                        <p class="text-gray-500">Welcome back, <?php echo htmlspecialchars($auth->getUser()['username']); ?></p>
                    </div>
                    <div class="flex items-center space-x-4">
                        <div class="flex items-center space-x-2">
                            <span class="status-dot w-3 h-3 rounded-full <?php echo $settings['is_active'] ? 'bg-green-500' : 'bg-red-500'; ?>"></span>
                            <span class="text-sm text-gray-600">
                                <?php echo $settings['is_active'] ? 'Bot Active' : 'Bot Inactive'; ?>
                            </span>
                        </div>
                        <div class="text-sm text-gray-500">
                            <?php echo $settings['is_test_mode'] ? '<span class="px-2 py-1 bg-yellow-100 text-yellow-700 rounded">Test Mode</span>' : '<span class="px-2 py-1 bg-green-100 text-green-700 rounded">Live Mode</span>'; ?>
                        </div>
                    </div>
                </div>
            </header>

            <!-- Content -->
            <div class="p-8">
                <!-- Stats Cards -->
                <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
                    <div class="bg-white rounded-xl shadow-sm p-6 card-hover">
                        <div class="flex items-center justify-between">
                            <div>
                                <p class="text-gray-500 text-sm">Total Profit</p>
                                <p class="text-2xl font-bold <?php echo ($totalProfit['total'] ?? 0) >= 0 ? 'text-green-600' : 'text-red-600'; ?>">
                                    <?php echo number_format($totalProfit['total'] ?? 0, 4); ?> USDT
                                </p>
                            </div>
                            <div class="w-12 h-12 bg-green-100 rounded-lg flex items-center justify-center">
                                <i class="fas fa-dollar-sign text-green-600 text-xl"></i>
                            </div>
                        </div>
                    </div>

                    <div class="bg-white rounded-xl shadow-sm p-6 card-hover">
                        <div class="flex items-center justify-between">
                            <div>
                                <p class="text-gray-500 text-sm">Win Rate</p>
                                <p class="text-2xl font-bold text-blue-600"><?php echo $winRate; ?>%</p>
                            </div>
                            <div class="w-12 h-12 bg-blue-100 rounded-lg flex items-center justify-center">
                                <i class="fas fa-percentage text-blue-600 text-xl"></i>
                            </div>
                        </div>
                    </div>

                    <div class="bg-white rounded-xl shadow-sm p-6 card-hover">
                        <div class="flex items-center justify-between">
                            <div>
                                <p class="text-gray-500 text-sm">Total Trades</p>
                                <p class="text-2xl font-bold text-purple-600"><?php echo $totalTrades['count'] ?? 0; ?></p>
                            </div>
                            <div class="w-12 h-12 bg-purple-100 rounded-lg flex items-center justify-center">
                                <i class="fas fa-exchange-alt text-purple-600 text-xl"></i>
                            </div>
                        </div>
                    </div>

                    <div class="bg-white rounded-xl shadow-sm p-6 card-hover">
                        <div class="flex items-center justify-between">
                            <div>
                                <p class="text-gray-500 text-sm">Active Trades</p>
                                <p class="text-2xl font-bold text-orange-600"><?php echo count($activeTrades); ?></p>
                            </div>
                            <div class="w-12 h-12 bg-orange-100 rounded-lg flex items-center justify-center">
                                <i class="fas fa-clock text-orange-600 text-xl"></i>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Charts & Tables Row -->
                <div class="grid grid-cols-1 lg:grid-cols-3 gap-8 mb-8">
                    <!-- Profit Chart -->
                    <div class="lg:col-span-2 bg-white rounded-xl shadow-sm p-6">
                        <h3 class="text-lg font-semibold text-gray-800 mb-4">Profit Overview</h3>
                        <canvas id="profitChart" height="100"></canvas>
                    </div>

                    <!-- Bot Status -->
                    <div class="bg-white rounded-xl shadow-sm p-6">
                        <h3 class="text-lg font-semibold text-gray-800 mb-4">Bot Status</h3>
                        <div class="space-y-4">
                            <div class="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
                                <span class="text-gray-600">Trading Pair</span>
                                <span class="font-semibold"><?php echo htmlspecialchars($settings['symbol'] ?? 'BTCUSDT'); ?></span>
                            </div>
                            <div class="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
                                <span class="text-gray-600">Timeframe</span>
                                <span class="font-semibold"><?php echo htmlspecialchars($settings['timeframe'] ?? '15m'); ?></span>
                            </div>
                            <div class="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
                                <span class="text-gray-600">Grid Trading</span>
                                <span class="px-2 py-1 rounded text-sm <?php echo $settings['mode_grid'] ? 'bg-green-100 text-green-700' : 'bg-gray-100 text-gray-600'; ?>">
                                    <?php echo $settings['mode_grid'] ? 'Enabled' : 'Disabled'; ?>
                                </span>
                            </div>
                            <div class="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
                                <span class="text-gray-600">DCA Trading</span>
                                <span class="px-2 py-1 rounded text-sm <?php echo $settings['mode_dca'] ? 'bg-green-100 text-green-700' : 'bg-gray-100 text-gray-600'; ?>">
                                    <?php echo $settings['mode_dca'] ? 'Enabled' : 'Disabled'; ?>
                                </span>
                            </div>
                            <div class="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
                                <span class="text-gray-600">Take Profit</span>
                                <span class="font-semibold"><?php echo $settings['take_profit_percentage'] ?? 2; ?>%</span>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Active Trades & Recent History -->
                <div class="grid grid-cols-1 lg:grid-cols-2 gap-8">
                    <!-- Active Trades -->
                    <div class="bg-white rounded-xl shadow-sm p-6">
                        <h3 class="text-lg font-semibold text-gray-800 mb-4">Active Trades</h3>
                        <?php if (empty($activeTrades)): ?>
                            <p class="text-gray-500 text-center py-8">No active trades</p>
                        <?php else: ?>
                            <div class="overflow-x-auto">
                                <table class="w-full">
                                    <thead>
                                        <tr class="text-left text-gray-500 text-sm border-b">
                                            <th class="pb-3">Symbol</th>
                                            <th class="pb-3">Side</th>
                                            <th class="pb-3">Price</th>
                                            <th class="pb-3">Qty</th>
                                            <th class="pb-3">Type</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($activeTrades as $trade): ?>
                                        <tr class="border-b last:border-0">
                                            <td class="py-3 font-medium"><?php echo htmlspecialchars($trade['symbol']); ?></td>
                                            <td class="py-3">
                                                <span class="px-2 py-1 rounded text-sm <?php echo $trade['side'] === 'buy' ? 'bg-green-100 text-green-700' : 'bg-red-100 text-red-700'; ?>">
                                                    <?php echo strtoupper($trade['side']); ?>
                                                </span>
                                            </td>
                                            <td class="py-3"><?php echo number_format($trade['entry_price'], 4); ?></td>
                                            <td class="py-3"><?php echo number_format($trade['quantity'], 4); ?></td>
                                            <td class="py-3">
                                                <span class="px-2 py-1 bg-blue-100 text-blue-700 rounded text-sm uppercase">
                                                    <?php echo $trade['trade_type']; ?>
                                                </span>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php endif; ?>
                    </div>

                    <!-- Recent Trades -->
                    <div class="bg-white rounded-xl shadow-sm p-6">
                        <h3 class="text-lg font-semibold text-gray-800 mb-4">Recent Trades</h3>
                        <?php if (empty($recentTrades)): ?>
                            <p class="text-gray-500 text-center py-8">No trade history</p>
                        <?php else: ?>
                            <div class="overflow-x-auto">
                                <table class="w-full">
                                    <thead>
                                        <tr class="text-left text-gray-500 text-sm border-b">
                                            <th class="pb-3">Symbol</th>
                                            <th class="pb-3">P/L</th>
                                            <th class="pb-3">Status</th>
                                            <th class="pb-3">Time</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($recentTrades as $trade): ?>
                                        <tr class="border-b last:border-0">
                                            <td class="py-3 font-medium"><?php echo htmlspecialchars($trade['symbol']); ?></td>
                                            <td class="py-3 <?php echo $trade['profit_loss'] >= 0 ? 'text-green-600' : 'text-red-600'; ?>">
                                                <?php echo number_format($trade['profit_loss'], 4); ?> USDT
                                            </td>
                                            <td class="py-3">
                                                <span class="px-2 py-1 rounded text-sm <?php echo $trade['status'] === 'win' ? 'bg-green-100 text-green-700' : 'bg-red-100 text-red-700'; ?>">
                                                    <?php echo ucfirst($trade['status']); ?>
                                                </span>
                                            </td>
                                            <td class="py-3 text-gray-500 text-sm">
                                                <?php echo date('H:i', strtotime($trade['created_at'])); ?>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Recent Logs -->
                <div class="mt-8 bg-white rounded-xl shadow-sm p-6">
                    <h3 class="text-lg font-semibold text-gray-800 mb-4">Recent Logs</h3>
                    <div class="space-y-2">
                        <?php foreach ($recentLogs as $log): ?>
                        <div class="flex items-center space-x-3 p-3 bg-gray-50 rounded-lg">
                            <span class="w-2 h-2 rounded-full <?php 
                                echo $log['level'] === 'error' ? 'bg-red-500' : 
                                     ($log['level'] === 'warning' ? 'bg-yellow-500' : 
                                     ($log['level'] === 'success' ? 'bg-green-500' : 'bg-blue-500')); 
                            ?>"></span>
                            <span class="text-sm text-gray-500"><?php echo date('H:i:s', strtotime($log['created_at'])); ?></span>
                            <span class="text-sm text-gray-700"><?php echo htmlspecialchars($log['message']); ?></span>
                        </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            </div>
        </main>
    </div>

    <script>
        // Profit Chart
        const ctx = document.getElementById('profitChart').getContext('2d');
        new Chart(ctx, {
            type: 'line',
            data: {
                labels: ['Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun'],
                datasets: [{
                    label: 'Profit (USDT)',
                    data: [12, 19, 15, 25, 22, 30, 35],
                    borderColor: '#10b981',
                    backgroundColor: 'rgba(16, 185, 129, 0.1)',
                    tension: 0.4,
                    fill: true
                }]
            },
            options: {
                responsive: true,
                plugins: {
                    legend: {
                        display: false
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        grid: {
                            color: 'rgba(0, 0, 0, 0.05)'
                        }
                    },
                    x: {
                        grid: {
                            display: false
                        }
                    }
                }
            }
        });
    </script>
</body>
</html>
