<?php
/**
 * User Management Page
 * Bitget Trading Bot - Admin Only
 */

require_once __DIR__ . '/../includes/Auth.php';
require_once __DIR__ . '/../includes/Database.php';

$auth = Auth::getInstance();
$auth->requireAdmin();

$db = Database::getInstance();

$message = '';
$error = '';

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['add_user'])) {
        $username = $_POST['username'] ?? '';
        $email = $_POST['email'] ?? '';
        $password = $_POST['password'] ?? '';
        $role = $_POST['role'] ?? 'user';

        if (empty($username) || empty($email) || empty($password)) {
            $error = 'Please fill all required fields';
        } else {
            $result = $auth->register($username, $email, $password, $role);
            if ($result['success']) {
                $message = 'User created successfully';
            } else {
                $error = $result['message'];
            }
        }
    }

    if (isset($_POST['delete_user'])) {
        $userId = $_POST['user_id'] ?? 0;
        $result = $auth->deleteUser($userId);
        if ($result['success']) {
            $message = 'User deleted successfully';
        } else {
            $error = $result['message'];
        }
    }

    if (isset($_POST['toggle_user'])) {
        $userId = $_POST['user_id'] ?? 0;
        $isActive = $_POST['is_active'] ?? 0;
        $result = $auth->updateUser($userId, ['is_active' => $isActive ? 0 : 1]);
        if ($result['success']) {
            $message = 'User status updated';
        } else {
            $error = $result['message'];
        }
    }
}

// Get all users
$users = $auth->getAllUsers();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>User Management - Bitget Trading Bot</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        @import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');
        body { font-family: 'Inter', sans-serif; }
        .gradient-bg { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); }
        .sidebar-link { transition: all 0.3s ease; }
        .sidebar-link:hover { background: rgba(99, 102, 241, 0.1); color: #6366f1; }
        .sidebar-link.active { background: rgba(99, 102, 241, 0.1); color: #6366f1; border-right: 3px solid #6366f1; }
    </style>
</head>
<body class="bg-gray-50">
    <div class="flex h-screen">
        <!-- Sidebar -->
        <aside class="w-64 bg-white shadow-xl">
            <div class="p-6">
                <div class="flex items-center space-x-3">
                    <div class="w-10 h-10 gradient-bg rounded-lg flex items-center justify-center">
                        <i class="fas fa-robot text-white text-xl"></i>
                    </div>
                    <div>
                        <h1 class="text-xl font-bold text-gray-800">Lompodoe</h1>
                        <p class="text-xs text-gray-500">Trading Bot</p>
                    </div>
                </div>
            </div>
            
            <nav class="mt-6">
                <a href="index.php" class="sidebar-link flex items-center px-6 py-3 text-gray-700">
                    <i class="fas fa-home w-6"></i>
                    <span>Dashboard</span>
                </a>
                <a href="trading.php" class="sidebar-link flex items-center px-6 py-3 text-gray-700">
                    <i class="fas fa-chart-line w-6"></i>
                    <span>Trading</span>
                </a>
                <a href="settings.php" class="sidebar-link flex items-center px-6 py-3 text-gray-700">
                    <i class="fas fa-cog w-6"></i>
                    <span>Settings</span>
                </a>
                <a href="history.php" class="sidebar-link flex items-center px-6 py-3 text-gray-700">
                    <i class="fas fa-history w-6"></i>
                    <span>History</span>
                </a>
                <a href="users.php" class="sidebar-link active flex items-center px-6 py-3 text-gray-700">
                    <i class="fas fa-users w-6"></i>
                    <span>Users</span>
                </a>
                <a href="logout.php" class="sidebar-link flex items-center px-6 py-3 text-gray-700">
                    <i class="fas fa-sign-out-alt w-6"></i>
                    <span>Logout</span>
                </a>
            </nav>
        </aside>

        <!-- Main Content -->
        <main class="flex-1 overflow-y-auto">
            <!-- Header -->
            <header class="bg-white shadow-sm">
                <div class="px-8 py-4">
                    <h2 class="text-2xl font-bold text-gray-800">User Management</h2>
                    <p class="text-gray-500">Manage bot users</p>
                </div>
            </header>

            <!-- Content -->
            <div class="p-8">
                <?php if ($message): ?>
                <div class="mb-6 p-4 bg-green-50 border border-green-200 rounded-lg flex items-center space-x-2">
                    <i class="fas fa-check-circle text-green-500"></i>
                    <span class="text-green-700"><?php echo htmlspecialchars($message); ?></span>
                </div>
                <?php endif; ?>

                <?php if ($error): ?>
                <div class="mb-6 p-4 bg-red-50 border border-red-200 rounded-lg flex items-center space-x-2">
                    <i class="fas fa-exclamation-circle text-red-500"></i>
                    <span class="text-red-700"><?php echo htmlspecialchars($error); ?></span>
                </div>
                <?php endif; ?>

                <!-- Add User Button -->
                <div class="mb-6">
                    <button onclick="document.getElementById('addUserModal').classList.remove('hidden')" 
                        class="px-6 py-2 bg-indigo-600 text-white rounded-lg hover:bg-indigo-700 transition-colors flex items-center space-x-2">
                        <i class="fas fa-plus"></i>
                        <span>Add New User</span>
                    </button>
                </div>

                <!-- Users Table -->
                <div class="bg-white rounded-xl shadow-sm overflow-hidden">
                    <table class="w-full">
                        <thead class="bg-gray-50">
                            <tr>
                                <th class="px-6 py-4 text-left text-sm font-medium text-gray-500">User</th>
                                <th class="px-6 py-4 text-left text-sm font-medium text-gray-500">Role</th>
                                <th class="px-6 py-4 text-left text-sm font-medium text-gray-500">Status</th>
                                <th class="px-6 py-4 text-left text-sm font-medium text-gray-500">Created</th>
                                <th class="px-6 py-4 text-left text-sm font-medium text-gray-500">Last Login</th>
                                <th class="px-6 py-4 text-left text-sm font-medium text-gray-500">Actions</th>
                            </tr>
                        </thead>
                        <tbody class="divide-y divide-gray-200">
                            <?php foreach ($users as $user): ?>
                            <tr class="hover:bg-gray-50">
                                <td class="px-6 py-4">
                                    <div class="flex items-center">
                                        <div class="w-10 h-10 bg-indigo-100 rounded-full flex items-center justify-center mr-3">
                                            <span class="text-indigo-600 font-semibold">
                                                <?php echo strtoupper(substr($user['username'], 0, 1)); ?>
                                            </span>
                                        </div>
                                        <div>
                                            <p class="font-medium text-gray-800"><?php echo htmlspecialchars($user['username']); ?></p>
                                            <p class="text-sm text-gray-500"><?php echo htmlspecialchars($user['email']); ?></p>
                                        </div>
                                    </div>
                                </td>
                                <td class="px-6 py-4">
                                    <span class="px-3 py-1 rounded-full text-sm <?php echo $user['role'] === 'admin' ? 'bg-purple-100 text-purple-700' : 'bg-blue-100 text-blue-700'; ?>">
                                        <?php echo ucfirst($user['role']); ?>
                                    </span>
                                </td>
                                <td class="px-6 py-4">
                                    <span class="px-3 py-1 rounded-full text-sm <?php echo $user['is_active'] ? 'bg-green-100 text-green-700' : 'bg-red-100 text-red-700'; ?>">
                                        <?php echo $user['is_active'] ? 'Active' : 'Inactive'; ?>
                                    </span>
                                </td>
                                <td class="px-6 py-4 text-gray-600">
                                    <?php echo date('M d, Y', strtotime($user['created_at'])); ?>
                                </td>
                                <td class="px-6 py-4 text-gray-600">
                                    <?php echo $user['last_login'] ? date('M d, Y H:i', strtotime($user['last_login'])) : 'Never'; ?>
                                </td>
                                <td class="px-6 py-4">
                                    <div class="flex items-center space-x-2">
                                        <form method="POST" action="" class="inline">
                                            <input type="hidden" name="user_id" value="<?php echo $user['id']; ?>">
                                            <input type="hidden" name="is_active" value="<?php echo $user['is_active']; ?>">
                                            <button type="submit" name="toggle_user" 
                                                class="p-2 <?php echo $user['is_active'] ? 'text-yellow-600 hover:bg-yellow-50' : 'text-green-600 hover:bg-green-50'; ?> rounded-lg transition-colors"
                                                title="<?php echo $user['is_active'] ? 'Deactivate' : 'Activate'; ?>">
                                                <i class="fas <?php echo $user['is_active'] ? 'fa-pause' : 'fa-play'; ?>"></i>
                                            </button>
                                        </form>
                                        
                                        <?php if ($user['id'] != $auth->getUserId()): ?>
                                        <form method="POST" action="" class="inline" onsubmit="return confirm('Are you sure you want to delete this user?');">
                                            <input type="hidden" name="user_id" value="<?php echo $user['id']; ?>">
                                            <button type="submit" name="delete_user" 
                                                class="p-2 text-red-600 hover:bg-red-50 rounded-lg transition-colors"
                                                title="Delete">
                                                <i class="fas fa-trash"></i>
                                            </button>
                                        </form>
                                        <?php endif; ?>
                                    </div>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </main>
    </div>

    <!-- Add User Modal -->
    <div id="addUserModal" class="hidden fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
        <div class="bg-white rounded-xl shadow-2xl w-full max-w-md mx-4">
            <div class="p-6 border-b">
                <div class="flex items-center justify-between">
                    <h3 class="text-xl font-bold text-gray-800">Add New User</h3>
                    <button onclick="document.getElementById('addUserModal').classList.add('hidden')" 
                        class="text-gray-400 hover:text-gray-600">
                        <i class="fas fa-times text-xl"></i>
                    </button>
                </div>
            </div>
            
            <form method="POST" action="" class="p-6">
                <div class="space-y-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Username</label>
                        <input type="text" name="username" required
                            class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500"
                            placeholder="Enter username">
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Email</label>
                        <input type="email" name="email" required
                            class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500"
                            placeholder="Enter email">
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Password</label>
                        <input type="password" name="password" required
                            class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500"
                            placeholder="Enter password">
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Role</label>
                        <select name="role" class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-indigo-500">
                            <option value="user">User</option>
                            <option value="admin">Admin</option>
                        </select>
                    </div>
                </div>
                
                <div class="mt-6 flex space-x-3">
                    <button type="button" onclick="document.getElementById('addUserModal').classList.add('hidden')" 
                        class="flex-1 px-4 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors">
                        Cancel
                    </button>
                    <button type="submit" name="add_user" 
                        class="flex-1 px-4 py-2 bg-indigo-600 text-white rounded-lg hover:bg-indigo-700 transition-colors">
                        Add User
                    </button>
                </div>
            </form>
        </div>
    </div>
</body>
</html>
