<?php
/**
 * Settings Page
 * Bitget Trading Bot
 */

require_once __DIR__ . '/../includes/Auth.php';
require_once __DIR__ . '/../includes/Database.php';

$auth = Auth::getInstance();
$auth->requireAuth();

$db = Database::getInstance();
$userId = $auth->getUserId();

$message = '';
$error = '';

// Get current settings
$settings = $db->fetch("SELECT * FROM trading_settings WHERE user_id = ?", [$userId]);
$apiSettings = $db->fetch("SELECT * FROM api_settings WHERE user_id = ? AND is_active = 1", [$userId]);
$telegramSettings = $db->fetch("SELECT * FROM telegram_settings WHERE user_id = ?", [$userId]);

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        // Update API Settings
        if (isset($_POST['update_api'])) {
            $apiData = [
                'api_key' => $_POST['api_key'] ?? '',
                'api_secret' => $_POST['api_secret'] ?? '',
                'passphrase' => $_POST['passphrase'] ?? '',
                'is_testnet' => isset($_POST['is_testnet']) ? 1 : 0
            ];

            if ($apiSettings) {
                $db->update('api_settings', $apiData, 'id = :id', ['id' => $apiSettings['id']]);
            } else {
                $apiData['user_id'] = $userId;
                $db->insert('api_settings', $apiData);
            }
            $message = 'API settings updated successfully';
        }

        // Update Telegram Settings
        if (isset($_POST['update_telegram'])) {
            $telegramData = [
                'bot_token' => $_POST['bot_token'] ?? '',
                'chat_id' => $_POST['chat_id'] ?? '',
                'enabled' => isset($_POST['telegram_enabled']) ? 1 : 0,
                'notify_trade' => isset($_POST['notify_trade']) ? 1 : 0,
                'notify_profit' => isset($_POST['notify_profit']) ? 1 : 0,
                'notify_loss' => isset($_POST['notify_loss']) ? 1 : 0,
                'notify_error' => isset($_POST['notify_error']) ? 1 : 0
            ];

            if ($telegramSettings) {
                $db->update('telegram_settings', $telegramData, 'id = :id', ['id' => $telegramSettings['id']]);
            } else {
                $telegramData['user_id'] = $userId;
                $db->insert('telegram_settings', $telegramData);
            }
            $message = 'Telegram settings updated successfully';
        }

        // Update Trading Settings
        if (isset($_POST['update_trading'])) {
            $tradingData = [
                'symbol' => $_POST['symbol'] ?? 'BTCUSDT',
                'timeframe' => $_POST['timeframe'] ?? '15m',
                'mode_grid' => isset($_POST['mode_grid']) ? 1 : 0,
                'mode_dca' => isset($_POST['mode_dca']) ? 1 : 0,
                'grid_levels' => intval($_POST['grid_levels'] ?? 10),
                'grid_investment' => floatval($_POST['grid_investment'] ?? 100),
                'grid_dynamic' => isset($_POST['grid_dynamic']) ? 1 : 0,
                'dca_levels' => intval($_POST['dca_levels'] ?? 5),
                'dca_drop_percentage' => floatval($_POST['dca_drop_percentage'] ?? 5),
                'dca_multiplier' => floatval($_POST['dca_multiplier'] ?? 1.5),
                'dca_max_investment' => floatval($_POST['dca_max_investment'] ?? 500),
                'take_profit_enabled' => isset($_POST['take_profit_enabled']) ? 1 : 0,
                'take_profit_percentage' => floatval($_POST['take_profit_percentage'] ?? 2),
                'risk_stop_loss_enabled' => isset($_POST['risk_stop_loss_enabled']) ? 1 : 0,
                'risk_stop_loss_percentage' => floatval($_POST['risk_stop_loss_percentage'] ?? 5),
                'risk_max_drawdown_enabled' => isset($_POST['risk_max_drawdown_enabled']) ? 1 : 0,
                'risk_max_drawdown_percentage' => floatval($_POST['risk_max_drawdown_percentage'] ?? 10),
                'risk_daily_loss_enabled' => isset($_POST['risk_daily_loss_enabled']) ? 1 : 0,
                'risk_daily_loss_amount' => floatval($_POST['risk_daily_loss_amount'] ?? 50),
                'risk_max_positions_enabled' => isset($_POST['risk_max_positions_enabled']) ? 1 : 0,
                'risk_max_positions' => intval($_POST['risk_max_positions'] ?? 5),
                'risk_trailing_stop_enabled' => isset($_POST['risk_trailing_stop_enabled']) ? 1 : 0,
                'risk_trailing_stop_percentage' => floatval($_POST['risk_trailing_stop_percentage'] ?? 3),
                'is_test_mode' => isset($_POST['is_test_mode']) ? 1 : 0,
                'test_balance' => floatval($_POST['test_balance'] ?? 10),
                'live_balance' => floatval($_POST['live_balance'] ?? 500)
            ];

            $db->update('trading_settings', $tradingData, 'user_id = :user_id', ['user_id' => $userId]);
            $message = 'Trading settings updated successfully';
        }

        // Toggle Bot Status
        if (isset($_POST['toggle_bot'])) {
            $newStatus = $settings['is_active'] ? 0 : 1;
            $db->update('trading_settings', ['is_active' => $newStatus], 'user_id = :user_id', ['user_id' => $userId]);
            
            // Log action
            $db->insert('bot_logs', [
                'user_id' => $userId,
                'level' => 'info',
                'message' => 'Bot ' . ($newStatus ? 'started' : 'stopped'),
                'context' => 'user_action'
            ]);
            
            $message = 'Bot ' . ($newStatus ? 'started' : 'stopped') . ' successfully';
            
            // Refresh settings
            $settings = $db->fetch("SELECT * FROM trading_settings WHERE user_id = ?", [$userId]);
        }

    } catch (Exception $e) {
        $error = $e->getMessage();
    }
}

// Refresh settings after update
$settings = $db->fetch("SELECT * FROM trading_settings WHERE user_id = ?", [$userId]);
$apiSettings = $db->fetch("SELECT * FROM api_settings WHERE user_id = ? AND is_active = 1", [$userId]);
$telegramSettings = $db->fetch("SELECT * FROM telegram_settings WHERE user_id = ?", [$userId]);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Settings - Bitget Trading Bot</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        @import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');
        body { font-family: 'Inter', sans-serif; }
        .gradient-bg { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); }
        .sidebar-link { transition: all 0.3s ease; }
        .sidebar-link:hover { background: rgba(99, 102, 241, 0.1); color: #6366f1; }
        .sidebar-link.active { background: rgba(99, 102, 241, 0.1); color: #6366f1; border-right: 3px solid #6366f1; }
        .tab-content { display: none; }
        .tab-content.active { display: block; }
    </style>
</head>
<body class="bg-gray-50">
    <div class="flex h-screen">
        <!-- Sidebar -->
        <aside class="w-64 bg-white shadow-xl">
            <div class="p-6">
                <div class="flex items-center space-x-3">
                    <div class="w-10 h-10 gradient-bg rounded-lg flex items-center justify-center">
                        <i class="fas fa-robot text-white text-xl"></i>
                    </div>
                    <div>
                        <h1 class="text-xl font-bold text-gray-800">Lompodoe</h1>
                        <p class="text-xs text-gray-500">Trading Bot</p>
                    </div>
                </div>
            </div>
            
            <nav class="mt-6">
                <a href="index.php" class="sidebar-link flex items-center px-6 py-3 text-gray-700">
                    <i class="fas fa-home w-6"></i>
                    <span>Dashboard</span>
                </a>
                <a href="trading.php" class="sidebar-link flex items-center px-6 py-3 text-gray-700">
                    <i class="fas fa-chart-line w-6"></i>
                    <span>Trading</span>
                </a>
                <a href="settings.php" class="sidebar-link active flex items-center px-6 py-3 text-gray-700">
                    <i class="fas fa-cog w-6"></i>
                    <span>Settings</span>
                </a>
                <a href="history.php" class="sidebar-link flex items-center px-6 py-3 text-gray-700">
                    <i class="fas fa-history w-6"></i>
                    <span>History</span>
                </a>
                <?php if ($auth->isAdmin()): ?>
                <a href="users.php" class="sidebar-link flex items-center px-6 py-3 text-gray-700">
                    <i class="fas fa-users w-6"></i>
                    <span>Users</span>
                </a>
                <?php endif; ?>
                <a href="logout.php" class="sidebar-link flex items-center px-6 py-3 text-gray-700">
                    <i class="fas fa-sign-out-alt w-6"></i>
                    <span>Logout</span>
                </a>
            </nav>
        </aside>

        <!-- Main Content -->
        <main class="flex-1 overflow-y-auto">
            <!-- Header -->
            <header class="bg-white shadow-sm">
                <div class="flex items-center justify-between px-8 py-4">
                    <div>
                        <h2 class="text-2xl font-bold text-gray-800">Settings</h2>
                        <p class="text-gray-500">Configure your trading bot</p>
                    </div>
                    <form method="POST" action="">
                        <button type="submit" name="toggle_bot" 
                            class="px-6 py-2 rounded-lg font-semibold flex items-center space-x-2 <?php echo $settings['is_active'] ? 'bg-red-500 hover:bg-red-600 text-white' : 'bg-green-500 hover:bg-green-600 text-white'; ?>">
                            <i class="fas <?php echo $settings['is_active'] ? 'fa-stop' : 'fa-play'; ?>"></i>
                            <span><?php echo $settings['is_active'] ? 'Stop Bot' : 'Start Bot'; ?></span>
                        </button>
                    </form>
                </div>
            </header>

            <!-- Content -->
            <div class="p-8">
                <?php if ($message): ?>
                <div class="mb-6 p-4 bg-green-50 border border-green-200 rounded-lg flex items-center space-x-2">
                    <i class="fas fa-check-circle text-green-500"></i>
                    <span class="text-green-700"><?php echo htmlspecialchars($message); ?></span>
                </div>
                <?php endif; ?>

                <?php if ($error): ?>
                <div class="mb-6 p-4 bg-red-50 border border-red-200 rounded-lg flex items-center space-x-2">
                    <i class="fas fa-exclamation-circle text-red-500"></i>
                    <span class="text-red-700"><?php echo htmlspecialchars($error); ?></span>
                </div>
                <?php endif; ?>

                <!-- Tabs -->
                <div class="bg-white rounded-xl shadow-sm">
                    <div class="border-b">
                        <nav class="flex">
                            <button onclick="showTab('api')" id="tab-api" class="tab-btn px-6 py-4 text-sm font-medium text-indigo-600 border-b-2 border-indigo-600">
                                <i class="fas fa-key mr-2"></i>API Settings
                            </button>
                            <button onclick="showTab('trading')" id="tab-trading" class="tab-btn px-6 py-4 text-sm font-medium text-gray-500 hover:text-gray-700">
                                <i class="fas fa-sliders-h mr-2"></i>Trading Settings
                            </button>
                            <button onclick="showTab('risk')" id="tab-risk" class="tab-btn px-6 py-4 text-sm font-medium text-gray-500 hover:text-gray-700">
                                <i class="fas fa-shield-alt mr-2"></i>Risk Management
                            </button>
                            <button onclick="showTab('telegram')" id="tab-telegram" class="tab-btn px-6 py-4 text-sm font-medium text-gray-500 hover:text-gray-700">
                                <i class="fab fa-telegram mr-2"></i>Telegram
                            </button>
                        </nav>
                    </div>

                    <!-- API Settings -->
                    <div id="content-api" class="tab-content active p-6">
                        <form method="POST" action="">
                            <input type="hidden" name="update_api" value="1">
                            
                            <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">API Key</label>
                                    <input type="text" name="api_key" value="<?php echo htmlspecialchars($apiSettings['api_key'] ?? ''); ?>"
                                        class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500">
                                </div>
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">API Secret</label>
                                    <input type="password" name="api_secret" value="<?php echo htmlspecialchars($apiSettings['api_secret'] ?? ''); ?>"
                                        class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500">
                                </div>
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Passphrase</label>
                                    <input type="password" name="passphrase" value="<?php echo htmlspecialchars($apiSettings['passphrase'] ?? ''); ?>"
                                        class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500">
                                </div>
                                <div class="flex items-center">
                                    <input type="checkbox" name="is_testnet" id="is_testnet" <?php echo ($apiSettings['is_testnet'] ?? 0) ? 'checked' : ''; ?>
                                        class="w-4 h-4 text-indigo-600 border-gray-300 rounded focus:ring-indigo-500">
                                    <label for="is_testnet" class="ml-2 text-sm text-gray-700">Use Testnet</label>
                                </div>
                            </div>

                            <div class="mt-6">
                                <button type="submit" class="px-6 py-2 bg-indigo-600 text-white rounded-lg hover:bg-indigo-700 transition-colors">
                                    Save API Settings
                                </button>
                            </div>
                        </form>
                    </div>

                    <!-- Trading Settings -->
                    <div id="content-trading" class="tab-content p-6">
                        <form method="POST" action="">
                            <input type="hidden" name="update_trading" value="1">

                            <!-- Mode Selection -->
                            <div class="mb-6">
                                <h4 class="text-lg font-medium text-gray-800 mb-4">Trading Mode</h4>
                                <div class="flex space-x-6">
                                    <label class="flex items-center">
                                        <input type="checkbox" name="mode_grid" <?php echo $settings['mode_grid'] ? 'checked' : ''; ?>
                                            class="w-4 h-4 text-indigo-600 border-gray-300 rounded focus:ring-indigo-500">
                                        <span class="ml-2 text-gray-700">Grid Trading</span>
                                    </label>
                                    <label class="flex items-center">
                                        <input type="checkbox" name="mode_dca" <?php echo $settings['mode_dca'] ? 'checked' : ''; ?>
                                            class="w-4 h-4 text-indigo-600 border-gray-300 rounded focus:ring-indigo-500">
                                        <span class="ml-2 text-gray-700">DCA (Averaging)</span>
                                    </label>
                                </div>
                            </div>

                            <!-- General Settings -->
                            <div class="grid grid-cols-1 md:grid-cols-3 gap-6 mb-6">
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Trading Pair</label>
                                    <input type="text" name="symbol" value="<?php echo htmlspecialchars($settings['symbol']); ?>"
                                        class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-indigo-500">
                                </div>
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Timeframe</label>
                                    <select name="timeframe" class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-indigo-500">
                                        <option value="1m" <?php echo $settings['timeframe'] === '1m' ? 'selected' : ''; ?>>1 Minute</option>
                                        <option value="5m" <?php echo $settings['timeframe'] === '5m' ? 'selected' : ''; ?>>5 Minutes</option>
                                        <option value="15m" <?php echo $settings['timeframe'] === '15m' ? 'selected' : ''; ?>>15 Minutes</option>
                                        <option value="30m" <?php echo $settings['timeframe'] === '30m' ? 'selected' : ''; ?>>30 Minutes</option>
                                        <option value="1h" <?php echo $settings['timeframe'] === '1h' ? 'selected' : ''; ?>>1 Hour</option>
                                        <option value="4h" <?php echo $settings['timeframe'] === '4h' ? 'selected' : ''; ?>>4 Hours</option>
                                        <option value="1d" <?php echo $settings['timeframe'] === '1d' ? 'selected' : ''; ?>>1 Day</option>
                                    </select>
                                </div>
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Take Profit (%)</label>
                                    <input type="number" step="0.01" name="take_profit_percentage" value="<?php echo $settings['take_profit_percentage']; ?>"
                                        class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-indigo-500">
                                </div>
                            </div>

                            <!-- Grid Settings -->
                            <div class="border-t pt-6 mb-6">
                                <h4 class="text-lg font-medium text-gray-800 mb-4">Grid Settings</h4>
                                <div class="grid grid-cols-1 md:grid-cols-3 gap-6">
                                    <div>
                                        <label class="block text-sm font-medium text-gray-700 mb-2">Grid Levels</label>
                                        <input type="number" name="grid_levels" value="<?php echo $settings['grid_levels']; ?>"
                                            class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-indigo-500">
                                    </div>
                                    <div>
                                        <label class="block text-sm font-medium text-gray-700 mb-2">Investment per Grid (USDT)</label>
                                        <input type="number" step="0.01" name="grid_investment" value="<?php echo $settings['grid_investment']; ?>"
                                            class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-indigo-500">
                                    </div>
                                    <div class="flex items-center pt-6">
                                        <input type="checkbox" name="grid_dynamic" id="grid_dynamic" <?php echo $settings['grid_dynamic'] ? 'checked' : ''; ?>
                                            class="w-4 h-4 text-indigo-600 border-gray-300 rounded focus:ring-indigo-500">
                                        <label for="grid_dynamic" class="ml-2 text-sm text-gray-700">Dynamic Grid</label>
                                    </div>
                                </div>
                            </div>

                            <!-- DCA Settings -->
                            <div class="border-t pt-6 mb-6">
                                <h4 class="text-lg font-medium text-gray-800 mb-4">DCA Settings</h4>
                                <div class="grid grid-cols-1 md:grid-cols-3 gap-6">
                                    <div>
                                        <label class="block text-sm font-medium text-gray-700 mb-2">DCA Levels</label>
                                        <input type="number" name="dca_levels" value="<?php echo $settings['dca_levels']; ?>"
                                            class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-indigo-500">
                                    </div>
                                    <div>
                                        <label class="block text-sm font-medium text-gray-700 mb-2">Drop Percentage (%)</label>
                                        <input type="number" step="0.01" name="dca_drop_percentage" value="<?php echo $settings['dca_drop_percentage']; ?>"
                                            class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-indigo-500">
                                    </div>
                                    <div>
                                        <label class="block text-sm font-medium text-gray-700 mb-2">Multiplier</label>
                                        <input type="number" step="0.01" name="dca_multiplier" value="<?php echo $settings['dca_multiplier']; ?>"
                                            class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-indigo-500">
                                    </div>
                                </div>
                            </div>

                            <!-- Balance Settings -->
                            <div class="border-t pt-6">
                                <h4 class="text-lg font-medium text-gray-800 mb-4">Balance Settings</h4>
                                <div class="grid grid-cols-1 md:grid-cols-3 gap-6">
                                    <div class="flex items-center">
                                        <input type="checkbox" name="is_test_mode" id="is_test_mode" <?php echo $settings['is_test_mode'] ? 'checked' : ''; ?>
                                            class="w-4 h-4 text-indigo-600 border-gray-300 rounded focus:ring-indigo-500">
                                        <label for="is_test_mode" class="ml-2 text-sm text-gray-700">Test Mode</label>
                                    </div>
                                    <div>
                                        <label class="block text-sm font-medium text-gray-700 mb-2">Test Balance (USDT)</label>
                                        <input type="number" step="0.01" name="test_balance" value="<?php echo $settings['test_balance']; ?>"
                                            class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-indigo-500">
                                    </div>
                                    <div>
                                        <label class="block text-sm font-medium text-gray-700 mb-2">Live Balance (USDT)</label>
                                        <input type="number" step="0.01" name="live_balance" value="<?php echo $settings['live_balance']; ?>"
                                            class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-indigo-500">
                                    </div>
                                </div>
                            </div>

                            <div class="mt-6">
                                <button type="submit" class="px-6 py-2 bg-indigo-600 text-white rounded-lg hover:bg-indigo-700 transition-colors">
                                    Save Trading Settings
                                </button>
                            </div>
                        </form>
                    </div>

                    <!-- Risk Management -->
                    <div id="content-risk" class="tab-content p-6">
                        <form method="POST" action="">
                            <input type="hidden" name="update_trading" value="1">

                            <div class="space-y-6">
                                <!-- Stop Loss -->
                                <div class="flex items-center justify-between p-4 bg-gray-50 rounded-lg">
                                    <div class="flex items-center">
                                        <input type="checkbox" name="risk_stop_loss_enabled" id="risk_stop_loss_enabled" 
                                            <?php echo $settings['risk_stop_loss_enabled'] ? 'checked' : ''; ?>
                                            class="w-4 h-4 text-indigo-600 border-gray-300 rounded focus:ring-indigo-500">
                                        <label for="risk_stop_loss_enabled" class="ml-3">
                                            <span class="block text-sm font-medium text-gray-700">Stop Loss</span>
                                            <span class="block text-xs text-gray-500">Stop trading when loss reaches</span>
                                        </label>
                                    </div>
                                    <div class="flex items-center space-x-2">
                                        <input type="number" step="0.01" name="risk_stop_loss_percentage" 
                                            value="<?php echo $settings['risk_stop_loss_percentage']; ?>"
                                            class="w-20 px-3 py-2 border border-gray-300 rounded-lg text-center">
                                        <span class="text-gray-500">%</span>
                                    </div>
                                </div>

                                <!-- Max Drawdown -->
                                <div class="flex items-center justify-between p-4 bg-gray-50 rounded-lg">
                                    <div class="flex items-center">
                                        <input type="checkbox" name="risk_max_drawdown_enabled" id="risk_max_drawdown_enabled" 
                                            <?php echo $settings['risk_max_drawdown_enabled'] ? 'checked' : ''; ?>
                                            class="w-4 h-4 text-indigo-600 border-gray-300 rounded focus:ring-indigo-500">
                                        <label for="risk_max_drawdown_enabled" class="ml-3">
                                            <span class="block text-sm font-medium text-gray-700">Max Drawdown</span>
                                            <span class="block text-xs text-gray-500">Stop when drawdown reaches</span>
                                        </label>
                                    </div>
                                    <div class="flex items-center space-x-2">
                                        <input type="number" step="0.01" name="risk_max_drawdown_percentage" 
                                            value="<?php echo $settings['risk_max_drawdown_percentage']; ?>"
                                            class="w-20 px-3 py-2 border border-gray-300 rounded-lg text-center">
                                        <span class="text-gray-500">%</span>
                                    </div>
                                </div>

                                <!-- Daily Loss Limit -->
                                <div class="flex items-center justify-between p-4 bg-gray-50 rounded-lg">
                                    <div class="flex items-center">
                                        <input type="checkbox" name="risk_daily_loss_enabled" id="risk_daily_loss_enabled" 
                                            <?php echo $settings['risk_daily_loss_enabled'] ? 'checked' : ''; ?>
                                            class="w-4 h-4 text-indigo-600 border-gray-300 rounded focus:ring-indigo-500">
                                        <label for="risk_daily_loss_enabled" class="ml-3">
                                            <span class="block text-sm font-medium text-gray-700">Daily Loss Limit</span>
                                            <span class="block text-xs text-gray-500">Maximum daily loss amount</span>
                                        </label>
                                    </div>
                                    <div class="flex items-center space-x-2">
                                        <input type="number" step="0.01" name="risk_daily_loss_amount" 
                                            value="<?php echo $settings['risk_daily_loss_amount']; ?>"
                                            class="w-24 px-3 py-2 border border-gray-300 rounded-lg text-center">
                                        <span class="text-gray-500">USDT</span>
                                    </div>
                                </div>

                                <!-- Max Positions -->
                                <div class="flex items-center justify-between p-4 bg-gray-50 rounded-lg">
                                    <div class="flex items-center">
                                        <input type="checkbox" name="risk_max_positions_enabled" id="risk_max_positions_enabled" 
                                            <?php echo $settings['risk_max_positions_enabled'] ? 'checked' : ''; ?>
                                            class="w-4 h-4 text-indigo-600 border-gray-300 rounded focus:ring-indigo-500">
                                        <label for="risk_max_positions_enabled" class="ml-3">
                                            <span class="block text-sm font-medium text-gray-700">Max Open Positions</span>
                                            <span class="block text-xs text-gray-500">Limit concurrent trades</span>
                                        </label>
                                    </div>
                                    <div class="flex items-center space-x-2">
                                        <input type="number" name="risk_max_positions" 
                                            value="<?php echo $settings['risk_max_positions']; ?>"
                                            class="w-20 px-3 py-2 border border-gray-300 rounded-lg text-center">
                                        <span class="text-gray-500">positions</span>
                                    </div>
                                </div>

                                <!-- Trailing Stop -->
                                <div class="flex items-center justify-between p-4 bg-gray-50 rounded-lg">
                                    <div class="flex items-center">
                                        <input type="checkbox" name="risk_trailing_stop_enabled" id="risk_trailing_stop_enabled" 
                                            <?php echo $settings['risk_trailing_stop_enabled'] ? 'checked' : ''; ?>
                                            class="w-4 h-4 text-indigo-600 border-gray-300 rounded focus:ring-indigo-500">
                                        <label for="risk_trailing_stop_enabled" class="ml-3">
                                            <span class="block text-sm font-medium text-gray-700">Trailing Stop</span>
                                            <span class="block text-xs text-gray-500">Lock in profits as price moves</span>
                                        </label>
                                    </div>
                                    <div class="flex items-center space-x-2">
                                        <input type="number" step="0.01" name="risk_trailing_stop_percentage" 
                                            value="<?php echo $settings['risk_trailing_stop_percentage']; ?>"
                                            class="w-20 px-3 py-2 border border-gray-300 rounded-lg text-center">
                                        <span class="text-gray-500">%</span>
                                    </div>
                                </div>
                            </div>

                            <div class="mt-6">
                                <button type="submit" class="px-6 py-2 bg-indigo-600 text-white rounded-lg hover:bg-indigo-700 transition-colors">
                                    Save Risk Settings
                                </button>
                            </div>
                        </form>
                    </div>

                    <!-- Telegram Settings -->
                    <div id="content-telegram" class="tab-content p-6">
                        <form method="POST" action="">
                            <input type="hidden" name="update_telegram" value="1">

                            <div class="grid grid-cols-1 md:grid-cols-2 gap-6 mb-6">
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Bot Token</label>
                                    <input type="text" name="bot_token" value="<?php echo htmlspecialchars($telegramSettings['bot_token'] ?? ''); ?>"
                                        class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-indigo-500"
                                        placeholder="123456789:ABCdefGHIjklMNOpqrsTUVwxyz">
                                </div>
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Chat ID</label>
                                    <input type="text" name="chat_id" value="<?php echo htmlspecialchars($telegramSettings['chat_id'] ?? ''); ?>"
                                        class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-indigo-500"
                                        placeholder="123456789">
                                </div>
                            </div>

                            <div class="mb-6">
                                <label class="flex items-center">
                                    <input type="checkbox" name="telegram_enabled" <?php echo ($telegramSettings['enabled'] ?? 0) ? 'checked' : ''; ?>
                                        class="w-4 h-4 text-indigo-600 border-gray-300 rounded focus:ring-indigo-500">
                                    <span class="ml-2 text-gray-700">Enable Telegram Notifications</span>
                                </label>
                            </div>

                            <div class="grid grid-cols-2 md:grid-cols-4 gap-4 mb-6">
                                <label class="flex items-center p-4 bg-gray-50 rounded-lg">
                                    <input type="checkbox" name="notify_trade" <?php echo ($telegramSettings['notify_trade'] ?? 1) ? 'checked' : ''; ?>
                                        class="w-4 h-4 text-indigo-600 border-gray-300 rounded focus:ring-indigo-500">
                                    <span class="ml-2 text-sm text-gray-700">Trades</span>
                                </label>
                                <label class="flex items-center p-4 bg-gray-50 rounded-lg">
                                    <input type="checkbox" name="notify_profit" <?php echo ($telegramSettings['notify_profit'] ?? 1) ? 'checked' : ''; ?>
                                        class="w-4 h-4 text-indigo-600 border-gray-300 rounded focus:ring-indigo-500">
                                    <span class="ml-2 text-sm text-gray-700">Profits</span>
                                </label>
                                <label class="flex items-center p-4 bg-gray-50 rounded-lg">
                                    <input type="checkbox" name="notify_loss" <?php echo ($telegramSettings['notify_loss'] ?? 1) ? 'checked' : ''; ?>
                                        class="w-4 h-4 text-indigo-600 border-gray-300 rounded focus:ring-indigo-500">
                                    <span class="ml-2 text-sm text-gray-700">Losses</span>
                                </label>
                                <label class="flex items-center p-4 bg-gray-50 rounded-lg">
                                    <input type="checkbox" name="notify_error" <?php echo ($telegramSettings['notify_error'] ?? 1) ? 'checked' : ''; ?>
                                        class="w-4 h-4 text-indigo-600 border-gray-300 rounded focus:ring-indigo-500">
                                    <span class="ml-2 text-sm text-gray-700">Errors</span>
                                </label>
                            </div>

                            <div class="mt-6">
                                <button type="submit" class="px-6 py-2 bg-indigo-600 text-white rounded-lg hover:bg-indigo-700 transition-colors">
                                    Save Telegram Settings
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </main>
    </div>

    <script>
        function showTab(tabName) {
            // Hide all tab contents
            document.querySelectorAll('.tab-content').forEach(content => {
                content.classList.remove('active');
            });
            
            // Remove active class from all tabs
            document.querySelectorAll('.tab-btn').forEach(btn => {
                btn.classList.remove('text-indigo-600', 'border-b-2', 'border-indigo-600');
                btn.classList.add('text-gray-500');
            });
            
            // Show selected tab content
            document.getElementById('content-' + tabName).classList.add('active');
            
            // Add active class to selected tab
            document.getElementById('tab-' + tabName).classList.remove('text-gray-500');
            document.getElementById('tab-' + tabName).classList.add('text-indigo-600', 'border-b-2', 'border-indigo-600');
        }
    </script>
</body>
</html>
